<?php
/**
 * Created by vm.pl
 * User: Rafał Ignaszewski (rafal@vm.pl)
 * Date: 13.12.13 11:30
 */

namespace VM\FinancialStatementsBundle\Services;

use Doctrine\ORM\EntityManager;
use VM\AdminBundle\Services\ApiGetter;
use VM\ApiBundle\Entity\AccountChange;
use VM\ApiBundle\Entity\BankServiceOptions;
use VM\ApiBundle\Entity\CompanyBankServiceOption;
use VM\ApiBundle\Entity\Company;
use VM\ApiBundle\Entity\CompanyDistributionSales;
use VM\ApiBundle\Entity\CompanyEmployee;
use VM\ApiBundle\Entity\CompanyAllegroCategoryItem;
use VM\ApiBundle\Entity\CompanyOffer;
use VM\ApiBundle\Entity\CompanyOfferReport;
use VM\ApiBundle\Entity\OfferReport;
use VM\ApiBundle\Entity\Round;
use VM\ApiBundle\Entity\SummaryCompanyRound;
use VM\FinancialStatementsBundle\Services\DataStructure\FinansowanieInwestycjiDS;
use VM\FinancialStatementsBundle\Services\DataStructure\FinansowanieProdukcjiDS;
use VM\FinancialStatementsBundle\Services\Gra;
use VM\ApiBundle\Entity\User;
use VM\FinancialStatementsBundle\Services\Restrictions\DataStructure\SegmentDS;
use VM\FinancialStatementsBundle\Services\Restrictions\DepartmentRestrictionsDS;
use VM\FinancialStatementsBundle\Services\Restrictions\SegmentRestrictions;

class FinancialStatementGame
{
    /**
     * @var EntityManager
     */
    protected $EntityManager;

    /**
     * @var \Symfony\Component\DependencyInjection\Container
     */
    protected $Container;

    /**
     * @var Round
     */
    protected $Round;

    /**
     * @var Company
     */
    protected $Company;

    /**
     * @var Gra
     */
    protected $game;

    /**
     * @var SegmentDS
     */
    public $SegmentDS;

    protected $WIBOR = 2.70;
    protected $procentAmortyzacji = 0.4;
    protected $wielkoscRynku = 2000000000;
    protected $udzialWRynku = 0.03;
    protected $liczbaZmianNaDzien = 3;
    protected $costManagement = 10000;

    protected $wielkoscRynkuOstatniejIteracji = 0;


    public function __construct(EntityManager $EntityManager,\Symfony\Component\DependencyInjection\Container $Container)
    {
        $this->wielkoscRynkuOstatniejIteracji = $this->wielkoscRynku;
        $this->EntityManager = $EntityManager;
        $this->Container = $Container;
        $this->WIBOR = $this->getApiGetter()->getActualWiborValue();
    }

    public function statement(Round $Round, Company $Company)
    {
        $this->Round = $Round;
        $this->Company = $Company;
        $this->game = new Gra($this->Round->getSequence());

        $this->setOldSummaryCompanyRoundsForGame();

        $this->prepareSegmentRestrictions();

        $this->setNewDataGame();

        $all = $this->game->getAll();
        $przeliczone = $this->evalAndGetNextIteration();

        $this->saveNewSummaryCompanyRoundsForGame($all);

        return $przeliczone;
    }

    public function getProductionData()
    {
        $data = $this->game->getProductionData();

        return array(
            'energyCost' => $this->getSurfaces() * $this->Company->getUserGroups()->getUser()->getMarketSegment()->getEnergyCost(),
            'price' => $this->getProductionPrice(),
            'selCount' => $data['sprzedano'],
            'cost' => $this->getKosztZmiennyMaterialowy(),
            'productionCount' => $data['wyprodukowano'],
            'incomeTax' => $data['podatekDochodowy'],
            'materialCostCorrection' => $data['zobowiazania'],
			'materialCostCorrectionOld' => $data['zobowiazaniaStare'],
            'incomeCorrection' => $data['naleznosci'],
            'incomeCorrectionOld' => $data['naleznosciStare']			
        );
    }

    public function getFullData()
    {
        return $this->game->getSprawozdania();
    }

    private function setNewDataGame()
    {
        $this->setFinansowanieInwestycji();
        $this->setPrzychodyFinansowe();
        $this->setKredytObrotowy();
        $this->setFinansowanieProdukcji();
    }

    private function prepareSegmentRestrictions()
    {
        /** @var SegmentRestrictions $SegmentRestrictions */
        $SegmentRestrictions = $this->Container->get('segment.restrictions');
        $MarketSegment = $this->Company->getUserGroups()->getUser()->getMarketSegment();

        $SegmentRestrictions->setData($MarketSegment,$this->Company);
        $SegmentRestrictions->generate();

        $this->SegmentDS = $SegmentRestrictions->getResults();
    }


    private function setFinansowanieInwestycji()
    {
        $typeSymbols = array('MOTO','KOMP','PHONE','OTHER');
        if(!$this->Company->getPlaces()->isEmpty())
        {
            /** @var CompanyAllegroCategoryItem $Place */
            foreach($this->Company->getPlaces() as $Place)
            {
                if($Place->getRoundId() == $this->Round->getId() && in_array($Place->getItem()->getType()->getSymbol(),$typeSymbols))
                {
                    $FinansowanieInwestycjiDS = new FinansowanieInwestycjiDS();
                    $FinansowanieInwestycjiDS->naklady = ($Place->getPrice()*$Place->getQuantity());

                    if(is_null($Place->getLoanId()))
                    {
                        $FinansowanieInwestycjiDS->czyKapitalWlasny = true;
                        $FinansowanieInwestycjiDS->kapitalWlasny = ($Place->getPrice()*$Place->getQuantity());
                        $FinansowanieInwestycjiDS->kapitalObcy = 0;
                        $FinansowanieInwestycjiDS->czyKredyt = false;
                        $FinansowanieInwestycjiDS->czyLeasing = false;
                        $FinansowanieInwestycjiDS->oprocentowanieKredytuLubLeasingu = 0.05;
                        $FinansowanieInwestycjiDS->okresKredytuLubLeasingu = 5;
                        $FinansowanieInwestycjiDS->ostatniaWplataLeasingu = 0;
                    }
                    else
                    {
                        $FinansowanieInwestycjiDS->czyKapitalWlasny = false;
                        $FinansowanieInwestycjiDS->kapitalWlasny = 0;
                        $FinansowanieInwestycjiDS->kapitalObcy = ($Place->getPrice()*$Place->getQuantity());
                        $FinansowanieInwestycjiDS->czyKredyt = true;
                        $FinansowanieInwestycjiDS->czyLeasing = false;
                        $FinansowanieInwestycjiDS->oprocentowanieKredytuLubLeasingu = $Place->getLoan()->getOption()->getPercent()/100;
                        $FinansowanieInwestycjiDS->okresKredytuLubLeasingu = $Place->getLoan()->getOption()->getDuration();
                        $FinansowanieInwestycjiDS->ostatniaWplataLeasingu = 0;
                    }

                    $FinansowanieInwestycjiDS->czyNaSrodkiTrwale = $Place->getPrice() > 3500;//wartość brutto
                    $FinansowanieInwestycjiDS->procentAmortyzacji = $Place->getDepreciation() / 100;

                    $this->game->addFinansowanieInwestycji($FinansowanieInwestycjiDS);
                }
            }
        }
    }

    private function setPrzychodyFinansowe()
    {
        $count = 0;
        $investmentPlus = 0;
        $investmentMinus = 0;
        if(!$this->Company->getBankServiceOptions()->isEmpty())
        {
            /** @var CompanyBankServiceOption $BankServiceOptions */
            foreach($this->Company->getBankServiceOptions() as $BankServiceOptions)
            {
                if($BankServiceOptions->getOption()->getBankService()->getSymbol() == 'INVESTMENT')
                {
                    if(($this->Round->getSequence() - $BankServiceOptions->getRound()->getSequence()) == ($BankServiceOptions->getOption()->getDuration() - 1))
                    {
                        $wibor = ($BankServiceOptions->getOption()->getWibor() == 1) ? $this->getApiGetter()->getActualWiborValue() : 0;
                        $interest = pow((1 + ($wibor + $BankServiceOptions->getOption()->getPercent())/100), $BankServiceOptions->getOption()->getDuration()) * $BankServiceOptions->getValue() - $BankServiceOptions->getValue();
                        $count += $interest;
                        if($BankServiceOptions->getOption()->getDuration() > 1)
                        {
//                            echo 'inv minus: '.$BankServiceOptions->getValue().'<br>';
                            $investmentMinus += $BankServiceOptions->getValue();
                        }
                        else
                        {
                            $investmentMinus += $BankServiceOptions->getValue();
                            $investmentPlus += $BankServiceOptions->getValue();
                        }
                    }
                    else
                    {
//                        echo 'inv plus: '.$BankServiceOptions->getValue().'<br>';
                        $investmentPlus += $BankServiceOptions->getValue();
                    }
                }
            }
        }

        $this->game->addPrzychodyFinansowe($count, $investmentPlus, $investmentMinus);
    }

    private function setKredytObrotowy()
    {
        $credit = array();
        $credit['value'] = 0;
        $credit['fullValue'] = 0;
        $credit['id'] = 0;
        $interest = 0;
        $ApiGetter = $this->getApiGetter();
        if(!$this->Company->getBankServiceOptions()->isEmpty())
        {
            /** @var CompanyBankServiceOption $CompanyBankServiceOption */
            foreach($this->Company->getBankServiceOptions() as $CompanyBankServiceOption)
            {
                if($CompanyBankServiceOption->getOption()->getBankService()->getSymbol() == 'REVOLVING_LOAN')
                {
                    $BankServiceOption = $CompanyBankServiceOption->getOption();
                    $wibor = ($BankServiceOption->getWibor()) ? $ApiGetter->getActualWiborValue() : 0;
                    $credit['value'] = $CompanyBankServiceOption->getValue();
                    $credit['id'] = $CompanyBankServiceOption->getId();
                    $credit['fullValue'] = $CompanyBankServiceOption->getValue();
                    $interest = $CompanyBankServiceOption->getValue() * ($wibor + $BankServiceOption->getPercent()) / 100;
                }
            }
        }

        $this->game->addKredytObrotowy($credit);
        $this->game->addOdsetkiKredytuObrotowego($interest);
    }

    private function setFinansowanieProdukcji()
    {
        $this->game->addFinansowanieProdukcji($this->createFinansowanieProdukcjiDS());
    }

    private function evalAndGetNextIteration()
    {
        return $this->game->evalAndGetNextIteration();
    }

    /**
     * @return FinansowanieProdukcjiDS
     */
    private function createFinansowanieProdukcjiDS()
    {
        $procentZmianyRynku = $this->Round->getMarket()/100;
        $PirBirBrakiForOffers = $this->getPirBirBrakiForOffers();

        $FinansowanieProdukcjiDS = new FinansowanieProdukcjiDS();

        $FinansowanieProdukcjiDS->wielkoscRynku = $this->wielkoscRynkuOstatniejIteracji + ($this->wielkoscRynkuOstatniejIteracji * $procentZmianyRynku);
        $FinansowanieProdukcjiDS->udzialWRynku = $this->udzialWRynku;
        $FinansowanieProdukcjiDS->cenaProduktu = $this->getProductionPrice();
        $FinansowanieProdukcjiDS->nakladyNaPiR = ($this->SegmentDS->maxPackages->getMaxEfficiency('PiR')+$this->getPrRFromDistributionSales()+$PirBirBrakiForOffers['PIR'])*$PirBirBrakiForOffers['xPIR'];
        $FinansowanieProdukcjiDS->nakladyNaInnowacje = ($this->SegmentDS->maxPackages->getMaxEfficiency('BiR')+$PirBirBrakiForOffers['BIR'])*$PirBirBrakiForOffers['xBIR'];
        $FinansowanieProdukcjiDS->obnizkaBrakow = $PirBirBrakiForOffers['BRAKI'];
        $FinansowanieProdukcjiDS->mozliwosciProdukcyjne =
            $this->SegmentDS->getMaxProductionForDay()
            * 21 //liczba dni pracujących na miesiac
            * 12 //liczba miesięcy
            ;
        $FinansowanieProdukcjiDS->planowanaProdukcja = $this->Company->getProductionAmount();
        $FinansowanieProdukcjiDS->kosztyStale = $this->getKosztyStale();
        $FinansowanieProdukcjiDS->kosztyZmienne = $this->getKosztyZmienne();
        $FinansowanieProdukcjiDS->bazowyKosztProdukcji = $this->getBazowyKosztProdukcji();

//        echo '<pre>';print_r($FinansowanieProdukcjiDS);
        return $FinansowanieProdukcjiDS;
    }

    private function getKosztyStale()
    {
        $kosztyStale = array();

        $kosztyStale['osobowe'] = array('q'=>1, 'kj'=>$this->getAllEmployeesCost()+$this->getCostManagement());
        $kosztyStale['materialowe'] = array('q'=>1, 'kj'=>0);
        $kosztyStale['energia'] = array('q'=>$this->getSurfaces(), 'kj'=>$this->Company->getUserGroups()->getUser()->getMarketSegment()->getEnergyCost() * 12);
        $kosztyStale['ubezpieczenia'] = array('q'=>1, 'kj'=>0);
        $kosztyStale['telekomunikacja'] = array('q'=>1, 'kj'=>0);
        $kosztyStale['podatkiLokalne'] = array('q'=>1, 'kj'=>0);
        $kosztyStale['oplatyBankowe'] = array('q'=>1, 'kj'=>$this->getOplatyBankowe());
        $kosztyStale['czynsz'] = array('q'=>1, 'kj'=>$this->getEstatePricesAll());
        $kosztyStale['uslugiObce'] = array('q'=>1, 'kj'=>$this->getCostsForOffers());

        return $kosztyStale;
    }

    private function getEstatePricesAll()
    {
        $priceAll = 0;
        $typeSymbols = array('OFFICE','DEPOT','STOCKROOM');
        if(!$this->Company->getPlaces()->isEmpty())
        {
            /** @var CompanyAllegroCategoryItem $Place */
            foreach($this->Company->getPlaces() as $Place)
            {
                if(in_array($Place->getItem()->getType()->getSymbol(),$typeSymbols))
                {
                    $priceAll += $Place->getPrice() * 12;
                }
            }
        }
        return $priceAll;
    }

    private function getAllEmployeesCost()
    {
        $count = 0;
        if(!$this->Company->getCompanyEmployees()->isEmpty())
        {
            foreach($this->Company->getCompanyEmployees() as $CompanyEmployees)
            {
                $count += $CompanyEmployees->getEmployee()->getSalary() * 12;
            }
        }
        return $count;
    }

    private function getCostManagement()
    {
        $rolesSymbol = array('M','C');

        $countManagement = $this->Company->getUserGroups()->getUsers()->filter(function($User) use($rolesSymbol) {
            $check = false;
            foreach($User->getCompanyRoles() as $CompanyRole)
            {
                if(in_array($CompanyRole->getSymbol(),$rolesSymbol)){$check = true;}
            }
            return $check;
        })->count();

        return $countManagement * $this->costManagement * 12;
    }

    private function getSurfaces()
    {
        $typeSymbols = array('OFFICE','DEPOT','STOCKROOM');
        $surfaces = 0;
        if(!$this->Company->getPlaces()->isEmpty())
        {
            /** @var CompanyAllegroCategoryItem $Place */
            foreach($this->Company->getPlaces() as $Place)
            {
                if(in_array($Place->getItem()->getType()->getSymbol(),$typeSymbols))
                {
                    $surfaces += $Place->getItem()->getSurface();
                }
            }
        }

        return $surfaces;
    }

    private function getOplatyBankowe()
    {
        $count = 0;
        if(!$this->Company->getBankServiceOptions()->isEmpty())
        {
            /** @var CompanyBankServiceOption $BankServiceOptions */
            foreach($this->Company->getBankServiceOptions() as $BankServiceOptions)
            {
                if($BankServiceOptions->getOption()->getBankService()->getSymbol() == 'CURRENT_ACCOUNT')
                {
                    $count += $BankServiceOptions->getOption()->getCost() * 12;
//                    $count += $BankServiceOptions->getOption()->getPercent() / 100 * $BankServiceOptions->getValue();
                }
            }
        }

        return $count;
    }

    private function getKosztyZmienne()
    {
        $kosztyZmienne = array();
        $kosztyZmienne['osobowe'] = array('kjz'=>0, 'cenaJednostki'=>0);
        $kosztyZmienne['materialowe'] = array(
            'kjz'=>1,
            'cenaJednostki'=> $this->getKosztZmiennyMaterialowy()
        );
        $kosztyZmienne['energia'] = array('kjz'=>0, 'cenaJednostki'=>0);
        $kosztyZmienne['woda'] = array('kjz'=>0, 'cenaJednostki'=>0);
        $kosztyZmienne['kosztyLogistyki'] = array('kjz'=>0, 'cenaJednostki'=>0);

        return $kosztyZmienne;
    }

    private function setOldSummaryCompanyRoundsForGame()
    {
        $SummaryCompanyRounds = $this->getSummaryCompanyRoundsForCompany();
        if(!empty($SummaryCompanyRounds))
        {
            foreach($SummaryCompanyRounds as $SummaryCompanyRound)
            {
                $data = json_decode($SummaryCompanyRound->getData(),true);
                if($data['numerIteracji']+1 != $this->Round->getSequence())
                {
                    $this->wielkoscRynkuOstatniejIteracji = $data['finansowanieProdukcji']['wielkoscRynku'];
                    $this->game->setAll($data);
                    $this->game->evalAndGetNextIteration();
                }
            }
        }
    }

    private function saveNewSummaryCompanyRoundsForGame($all)
    {
        $dataProduction = $this->game->getProductionData();

        $SummaryCompanyRound = $this->getSummaryCompanyRoundsForCompanyAndRound();
        if(empty($SummaryCompanyRound))
        {
            $SummaryCompanyRound = new SummaryCompanyRound();
            $SummaryCompanyRound->setCompany($this->Company);
            $SummaryCompanyRound->setRound($this->Round);
            $SummaryCompanyRound->setZapasy($dataProduction['zapasy']);
        }

        $SummaryCompanyRound->setData(json_encode($all));
        $SummaryCompanyRound->setResults(json_encode($this->game->getSprawozdania()));

        $this->EntityManager->persist($SummaryCompanyRound);
        $this->EntityManager->flush();
        $this->EntityManager->refresh($SummaryCompanyRound);
    }

    private function getSummaryCompanyRoundsForCompany()
    {
        return $this->EntityManager->getRepository('VM\ApiBundle\Entity\SummaryCompanyRound')->findBy(array('companyId'=>$this->Company->getId()));
    }

    private function getSummaryCompanyRoundsForCompanyAndRound()
    {
        return $this->EntityManager->getRepository('VM\ApiBundle\Entity\SummaryCompanyRound')->findOneBy(array('companyId'=>$this->Company->getId(),'roundId'=>$this->Round->getId()));
    }

    private function getSummaryCompanyRoundsForCompanyAndPreviewRound()
    {
        $PreviewRound = $this->EntityManager->getRepository('VMApiBundle:Round')->findOneBy(array('sequence'=>$this->Round->getSequence()-1));
        if(empty($PreviewRound))
        {
            return false;
        }
        return $this->EntityManager->getRepository('VM\ApiBundle\Entity\SummaryCompanyRound')->findOneBy(array('companyId'=>$this->Company->getId(),'roundId'=>$PreviewRound->getId()));
    }

    private function getBazowyKosztProdukcji()
    {
        $bazowyKosztProdukcji = $this->Company->getUserGroups()->getUser()->getMarketSegment()->getProductionCost();
        $currencyProductionCost = $this->Company->getUserGroups()->getUser()->getMarketSegment()->getCurrencyProductionCost();

        $GpwSession = $this->EntityManager->getRepository('ImportApiBundle:GpwSession')->getActualSessionForIndex($currencyProductionCost);
        if(!empty($GpwSession))
        {
            $bazowyKosztProdukcji *= $GpwSession->getClose();
        }

        return $bazowyKosztProdukcji;
    }

    private function getKosztZmiennyMaterialowy()
    {
        $kosztZmiennyMaterialowy = $this->getBazowyKosztProdukcji();
        $kosztZmiennyMaterialowy += $kosztZmiennyMaterialowy * $this->Company->getChangeProductionCost() /100;

        return $kosztZmiennyMaterialowy;
    }

    private function getPirBirBrakiForOffers()
    {
        $return = array('BIR'=>0,'PIR'=>0,'xBIR'=>0,'xPIR'=>0,'BRAKI'=>0);

        if(!$this->Company->getRoundOffers()->isEmpty())
        {
            /** @var CompanyOffer $RoundOffer */
            foreach($this->Company->getRoundOffers() as $RoundOffer)
            {
                $efficiency = json_decode($RoundOffer->getOffer()->getEfficiency(),true);
                if(!empty($efficiency))
                {
                    $intervalSequenceRound = $this->Round->getSequence() - $RoundOffer->getRound()->getSequence();
                    foreach($efficiency as $key => $value)
                    {
                        $value = explode('|',$value);
                        if(isset($value[$intervalSequenceRound]))
                        {
                            if($key == 'BRAKI')
                            {
                                $diff = 100 - $return[$key];
                                $return[$key] = $return[$key] + $diff*($value[$intervalSequenceRound]/100);
//                                $return[$key] += $value[$intervalSequenceRound];
                            }
                            else
                            {
                                $return[$key] += $value[$intervalSequenceRound];
                            }
                        }
                    }
                }
            }
        }
        $return['BRAKI'] = ($return['BRAKI'] > 100) ? 100 : $return['BRAKI'];
        $return['BRAKI'] = ($return['BRAKI'] < 0) ? 0 : $return['BRAKI'];
        $return['xBIR'] = ($return['xBIR'] == 0) ? 1 : $return['xBIR'];
        $return['xPIR'] = ($return['xPIR'] == 0) ? 1 : $return['xPIR'];
        return $return;
    }

    public function getProductionPrice()
    {
        $productionPrice = $this->Company->getProductionPrice();// + $this->Company->getProductionPrice();// * $this->Company->getChangeProductionCost()/100;
        $productionAmount = $this->Company->getProductionAmount();
        $SummaryCompanyRoundsForCompany = $this->getSummaryCompanyRoundsForCompanyAndPreviewRound();
        $productionAmount += ($SummaryCompanyRoundsForCompany) ? $SummaryCompanyRoundsForCompany->getZapasy() : 0;
        if($this->Company->getUserGroups()->getUser()->getMarketSegment()->getDistributionSales()->isEmpty() || $this->Company->getCompanyDistributionSales()->isEmpty() || $productionPrice <= 0 || $productionAmount <= 0)
        {
            return $productionPrice;
        }
        $count = 0;

        /** @var CompanyDistributionSales $CompanyDistributionSales */
        foreach($this->Company->getCompanyDistributionSales() as $CompanyDistributionSales)
        {
            $prod = $productionAmount * $CompanyDistributionSales->getValue()/100;
            $prodPrice = $productionPrice - $productionPrice * $CompanyDistributionSales->getDistributionSales()->getDiscount()/100;
            $count += $prod * $prodPrice;
        }
        return $count/$productionAmount;
    }

    private function getPrRFromDistributionSales()
    {
        $pir = 0;
        if($this->Company->getUserGroups()->getUser()->getMarketSegment()->getDistributionSales()->isEmpty() || $this->Company->getCompanyDistributionSales()->isEmpty())
        {
            return $pir;
        }
        /** @var CompanyDistributionSales $CompanyDistributionSales */
        foreach($this->Company->getCompanyDistributionSales() as $CompanyDistributionSales)
        {
            $efficiency = json_decode($CompanyDistributionSales->getDistributionSales()->getEfficiency(),true);
            if(isset($efficiency['PIR']))
            {
                $efficiencyPIR = explode('|',$efficiency['PIR']);
                $pirForDistribution = $CompanyDistributionSales->getDistributionSales()->getDiscount() * $efficiencyPIR[0]/100 * $CompanyDistributionSales->getValue()/100;
                $pir += $pirForDistribution;
            }
        }
        return $pir;
    }

    private function getCostsForOffers()
    {
        $cost = 0;
        if(!$this->Company->getRoundOffers()->isEmpty())
        {
            /** @var CompanyOffer $RoundOffer */
            foreach($this->Company->getRoundOffers() as $RoundOffer)
            {
//                echo $RoundOffer->getRound()->getSequence()." ".$this->Round->getSequence()."<br>";
                if($RoundOffer->getRound()->getSequence() == $this->Round->getSequence())
                {
                    $price = json_decode($RoundOffer->getOffer()->getPrice(),true);
                    if(!empty($price))
                    {
                        $cost += $price;
                    }
                }
            }
        }

        if(!$this->Company->getCompanyOfferReports()->isEmpty())
        {
            /** @var CompanyOfferReport $CompanyOfferReport */
            foreach($this->Company->getCompanyOfferReports() as $CompanyOfferReport)
            {
                if($CompanyOfferReport->getRound()->getSequence() == $this->Round->getSequence())
                {
                    $price = json_decode($CompanyOfferReport->getReport()->getPrice(),true);
                    if(!empty($price))
                    {
                        $cost += $price;
                    }
                }
            }
        }

        return $cost;
    }

    /**
     * @return ApiGetter
     */
    protected function getApiGetter()
    {
        return $this->Container->get('vm_admin.api.getter');
    }
}